/* Filename : binco.h
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : Binomial coefficients functions.
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 *
 * Remarks  : ISO conform (-ansi -pedantic)
 */

/* ------------------------------------------------------------------------ */

#ifndef _BINCO_H
#define _BINCO_H

/* ------------------------------------------------------------------------ */

/* Error codes */
#define BINCO_NOERROR       0
#define BINCO_ERR_INTERN    1
#define BINCO_OVERFLOW      2
#define BINCO_ERR_MEMLACK   3

/* ------------------------------------------------------------------------ */

/* Function : int binco_init(int nm)
 * Params   : The maximum known sample size.
 * Return   : Error / Warning code
 * Goal     : Preallocate the binomial coefficient matrix if you already
 *            know the maximum sample size. This is not essential but
 *            improve performance if you are going to have something like
 *            incremental sample sizes.
 * Remarks  : If matrix is already allocated, this function will first free
 *            it before rebuild. That is you do not have to call the
 *            matrix deallocation function before calling this function.
 *
 * PreCond  : NONE
 * PostCond : Return error code. Memory usage may increase.
 */
int binco_init(int nmax);

/* ------------------------------------------------------------------------ */

/* Function : int binco_maxsize(void)
 * Params   : NONE
 * Return   : Error / Warning code
 * Goal     : Return current maximum coefficient for binomial generation
 *
 * PreCond  : NONE
 * PostCond : Return maximum size.
 */

int binco_maxsize(void);

/* ------------------------------------------------------------------------ */

/* Function : long double binco_get(int n, int k)
 * Params   : C(n,k)
 * Return   : Error / Warning code
 * Goal     : Return binomial coefficient C(n,k)
 *
 * PreCond  : NONE
 * PostCond : Return error code. Memory usage may increase.
 */

long double binco_get(int n, int k);

/* ------------------------------------------------------------------------ */

/* Function : int binco_free(void)
 * Params   : NONE
 * Return   : Error / Warning code
 * Goal     : Free binomial coefficient matrix
 *
 * PreCond  : NONE
 * PostCond : Return error code. Memory usage may decrease.
 */

int binco_free(void);

/* ------------------------------------------------------------------------ */

#endif

/* ------------------------------------------------------------------------ */

