/* Filename : fisher.h
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : This is the implementation of the Fisher non-parametric
 *            test.
 *
 *            Online implementation and project description available
 *            at http://qualopt.eivd.ch
 *
 * Warnings : The function generates a coefficient matrix for binomial
 *            computations. This matrix has a size MAX(n,m) given to the
 *            fisher(...) function. Matrix is remanent between two function
 *            call to avoid constant recomputation of binomial coefficients.
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 *
 * Compil.  : Please see Makefile
 * Remarks  : ISO conform (-ansi -pedantic)
 */


/* = INCLUSION CONTROL ==================================================== */


#ifndef _FISHER_H
#define _FISHER_H


/* = ERROR / WARNING CODES ================================================ */


/* No error */
#define FISHER_NOERROR             0

/* Bad function parameters */
#define FISHER_BAD_PARAMETERS      1

/* Binomial coefficient overflow */
#define FISHER_BINCO_ERROR         2

/* Internal error */
#define FISHER_ERR_INTERN          3


/* = EXPORTED FUNCTIONS =================================================== */


/* Function : int fisher_uni(int a, int n, int b, int m, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the Fisher test according to given values
 *            (Unilateral Test)
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : S contains the result. Return error code. Memory usage may
 *            increase.
 */

int fisher_uni(int a, int n, int b, int m, double* S);

/* ------------------------------------------------------------------------ */

/* Function : int fisher_bi(int a, int n, int b, int m, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the Fisher test according to given values
 *            (Bilateral Test)
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : S contains the result. Return error code. Memory usage may
 *            increase.
 */

int fisher_bi(int a, int n, int b, int m, double* S);

/* ------------------------------------------------------------------------ */

/* Function : char* fisher_err2str(int error_code)
 * Params   : The error / warning code
 * Return   : Error / Warning string description, NULL if error code unknown.
 * Goal     : Get the description of an error code
 *
 * PreCond  : NONE
 * PostCond : Return error descrition, NULL if error code unknown
 */

char* fisher_err2str(int error_code);

/* ======================================================================== */

#endif

/* ======================================================================== */

