/* Filename : taillard.h
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : A Statistical Test for Comparing Success Rates.
 *
 *            This is the implementation of the statistical test presented
 *            in the paper 'Statistical Test for Comparing Success Rates'
 *            written by Eric D. Taillard, INA-EiVD.
 *
 *            Online implementation and project description available
 *            at http://qualopt.eivd.ch
 *
 * Warnings : The function generates a coefficient matrix for binomial
 *            computations. This matrix has a size MAX(n,m) given to the
 *            taillard(...) function. Matrix is remanent between two function
 *            call to avoid constant recomputation of binomial coefficients.
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 *
 * Compil.  : Please see Makefile
 * Remarks  : ISO conform (-ansi -pedantic)
 */


/* = INCLUSION CONTROL ==================================================== */


#ifndef _STCSR_H
#define _STCSR_H


/* = ERROR / WARNING CODES ================================================ */


/* No error */
#define STCSR_NOERROR               0

/* Bad function parameters */
#define STCSR_BAD_PARAMETERS        1

/* Binomial coefficient overflow */
#define STCSR_BINCO_ERROR           2

/* Internal error */
#define STCSR_ERR_INTERN            3


/* = EXPORTED FUNCTIONS =================================================== */


/* Function : int taillard_uni(int a, int n, int b, int m, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the probability S to observe 'a' successes or more in
 *            the first population (of size n) and 'b' successes or less in
 *            the second population (of size m)
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : S contains the result. Return error code. Memory usage may
 *            increase.
 */

int taillard_uni(int a, int n, int b, int m, double* p, double* S);

/* ------------------------------------------------------------------------ */

/* Function : int taillard_bi(int a, int n, int b, int m, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the probability S to observe that the success rate
 *            of both samples are different
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : T contains the result. Return error code. Memory usage may
 *            increase.
 */

int taillard_bi(int a, int n, int b, int m, double* p, double* T);

/* ------------------------------------------------------------------------ */

/* Function : char* taillard_err2str(int error_code)
 * Params   : The error / warning code
 * Return   : Error / Warning string description, NULL if error code unknown.
 * Goal     : Get the description of an error code
 *
 * PreCond  : NONE
 * PostCond : Return error descrition, NULL if error code unknown
 */

char* taillard_err2str(int error_code);

/* ======================================================================== */

#endif

