/* Filename : taillard_table.c
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : Test the implementation of the 'Statistical Test for Comparing
 *            Success Rates' (STATS Library).
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 */

/* ------------------------------------------------------------------------ */

/* Standard librairies */
#include <stdio.h>
#include <stdlib.h>

/* ------------------------------------------------------------------------ */

/* Statistical test */
#include <taillard.h>

/* Binomial coefficients */
#include <binco.h>

/* ------------------------------------------------------------------------ */

/* Precision for result */
#define PRECISION 6

/* ------------------------------------------------------------------------ */

/* Usage */
void usage(char* prog)
{
    fprintf(stderr, "\n\nUsage : %s [ n m c ]\n\n", prog);
    fprintf(stderr, "\tn : N value (Sample 1) >= 2\n"
                    "\tm : M value (Sample 2) >= 2\n"
                    "\tc : Confidence level [%%]\n\n");
}

/* ------------------------------------------------------------------------ */

void output(int n, int m, double c,
        int (*taillard_func)(int, int, int, int, double*, double*))
{
    /* Loop indices */
    int a, na, b, mb;

    /* To keep limit accepted rates when the table is generated */
    int star = 0, at_least_one, astar = 0, bstar = 0;
    double Sstar = 0.0;

    /* A current probability */
    double S, p;

    /* Let's make all permutations */
    for (na = 2; na <= n; na++)
    {
        for (mb = 2; mb <= m; mb++)
        {
            printf("{%d,%d}\t", na, mb);
            at_least_one = 0;
            Sstar = 1.01;

            for (a = 0; a <= na; a++)
            {
                Sstar = 1.01;

                for (b = 0; b <= mb; b++)
                {
                    if ((a == 0 && b == 0) || (a == na && b == mb) ||
                            (a * mb < b * na))
                        continue;

                    taillard_func(a, na, b, mb, &p, &S);

                    if (S >= c && S < Sstar)
                    {
                        astar = a; bstar = b; Sstar = S;
                        star = 1;
                    }
                }

                /* Do we found a > (the best) Confidence Level */
                if (star)
                {
                    printf("\t[%d,%d]", astar, bstar);
                    star = 0; at_least_one = 1;
                }
            }
            if (!at_least_one)
            {
                printf("---");
            }
            printf("\n");
        }
        printf("\n");
    }
}

/* ------------------------------------------------------------------------ */

/* Main... */
int main(int argc, char** argv)
{
    /* Maximum n and m values */
    int n, m;

    /* Confidence level */
    double c;

    /* If no arguments given, assume hand input */
    if (argc < 2)
    {
        /* Parameters entries */
        printf("\nEnter n : "); fflush(stdout);
        scanf("%d", &n);

        printf("Enter m : "); fflush(stdout);
        scanf("%d", &m);
        printf("\n");

        printf("Enter confidence level [%%] : "); fflush(stdout);
        scanf("%lf", &c);
        printf("\n");
    }
    /* If enough arguments given, assume command line parameters */
    else if (argc == 4)
    {
        n = (int)strtol(argv[1], (char**)NULL, 10);
        m = (int)strtol(argv[2], (char**)NULL, 10);
        c = strtod(argv[3], (char**)NULL);
    }
    /* Else consider a explanation */
    else
    {
        usage(argv[0]);
        exit(EXIT_FAILURE);
    }

    /* Adjust confidence level */
    c /= 100.0;

    /* Check parameters */
    if (n < 2 || m < 2)
    {
        fprintf(stderr, "\nError : n and m must be >= 2");
        usage(argv[0]);
        exit(EXIT_FAILURE);
    }

    /* Preallocate binomial matrix */
    binco_init(n > m ? n : m);

    /* Print little explanation in the beginning of ouput */
    printf("Output format :\n\n");
    printf("{n,m}\n"
           "\t[a,b] ...     (Limit couple to be >= Confidence level)\n\n");

    printf("ONE-TAILED TEST\n\n");
    output(n, m, c, taillard_uni);

    printf("\n");

    printf("TWO-TAILED TEST\n\n");
    output(n, m, c, taillard_bi);

    /* Free binomial matrix */
    binco_free();

    /* Exit... */
    exit(EXIT_SUCCESS);
}

/* ------------------------------------------------------------------------ */

