/* Filename : taillard_test.c
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : Test the implementation of the 'Statistical Test for Comparing
 *            Success Rates' (STATS Library).
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 */

/* ------------------------------------------------------------------------ */

/* Standard librairies */
#include <stdio.h>
#include <stdlib.h>

/* ------------------------------------------------------------------------ */

/* Statistical test */
#include <taillard.h>

/* Binomial coefficient */
#include <binco.h>

/* ------------------------------------------------------------------------ */

/* Precision for result */
#define PRECISION 6

/* ------------------------------------------------------------------------ */

/* Prints welcome message */
void print_welcome(void)
{
    printf("\nA Statistical Test to Compare Success Rates\n");
    printf("===========================================\n");
    printf(" Let the sample 1 have a success rate of a/n\n");
    printf(" and sample 2 have a success rate of b/m :\n");
}

/* ------------------------------------------------------------------------ */

/* Main... */
int main(int argc, char** argv)
{
    /* Loop indices */
    int a, b;

    /* Maximum n and m values */
    int n, m;

    /* Error code */
    int error_code;

    /* A current probability */
    double S, p;

    /* Prints welcome message */
    print_welcome();

    /* Parameters entries */
    printf("\nEnter a : "); fflush(stdout);
    scanf("%d", &a);

    printf("Enter n : "); fflush(stdout);
    scanf("%d", &n);
    printf("\n");

    printf("Enter b : "); fflush(stdout);
    scanf("%d", &b);

    printf("Enter m : "); fflush(stdout);
    scanf("%d", &m);
    printf("\n");

    /* Compute S / Unilateral */
    if ((error_code = taillard_uni(a, n, b, m, &p, &S) != STCSR_NOERROR))
    {
        fprintf(stderr, "\n%s\n\n", taillard_err2str(error_code));
        exit(EXIT_FAILURE);
    }

    /* Print S */
    printf("[ Taillard Test :: One-Tailed Test ]\n\n");
    printf("The p-value (significance level) associated with null "
            "hypothesis\n"
            "(both proportions identical to p=%.*f) is %.*f. Confidence\n"
            "level of alternate hypothesis (sample 1 has a success rate\n"
            "higher than sample 2) : %.*f\n\n",
           PRECISION, p, PRECISION, S, PRECISION, 1.0 - S);

    /* Compute S / Bilateral */
    if ((error_code = taillard_bi(a, n, b, m, &p, &S) != STCSR_NOERROR))
    {
        fprintf(stderr, "\n%s\n\n", taillard_err2str(error_code));
        exit(EXIT_FAILURE);
    }

    /* Print T */
    printf("[ Taillard Test :: Two-Tailed Test ]\n\n");
    printf("The p-value (significance level) associated with null "
            "hypothesis\n"
            "(both proportions identical to p=%.*f) is %.*f. Confidence\n"
            "level of alternate hypothesis (sample 1 has a success rate\n"
            "different to sample 2) : %.*f\n\n",
           PRECISION, p, PRECISION, S, PRECISION, 1.0 - S);


    /* Free binomial matrix */
    binco_free();

    /* Exit... */
    exit(EXIT_SUCCESS);
}

/* ------------------------------------------------------------------------ */

