/* Filename : binco.c
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : Binomial coefficients functions.
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 *
 * Remarks  : ISO conform (-ansi -pedantic)
 */

/* ------------------------------------------------------------------------ */

#include "binco.h"

/* ------------------------------------------------------------------------ */

#include <stdlib.h>
#include <math.h>

/* ------------------------------------------------------------------------ */

/* Current binomail matrix size */
static int binco_size = -1;

/* Binomials matrix */
static long double* binco_matrix = NULL;

/* ------------------------------------------------------------------------ */

/* Function : int binco_maxsize(void)
 * Params   : NONE
 * Return   : Error / Warning code
 * Goal     : Return current maximum coefficient for binomial generation
 *
 * PreCond  : NONE
 * PostCond : Return maximum size.
 */

int binco_maxsize(void)
{
    return binco_size;
}

/* ------------------------------------------------------------------------ */

/* Generate binomial coefficients matrix */
static int binco_gen(void)
{
    /* Loop indices */
    int i, j;

    /* Check matrix allocation */
    if (binco_matrix == NULL)
    {
        return BINCO_ERR_INTERN;
    }

    /* Generate extremum value of Pascal triangle */
    for (i = 0; i < binco_size; i++)
    {
        binco_matrix[i * binco_size] = 1.0;
        binco_matrix[i * binco_size + i] = 1.0;
    }

    /* Generate Pascal triangle coefficients */
    for (i = 2; i < binco_size; i++)
    {
        for (j = 1; j <= i - 1; j++)
        {
            binco_matrix[i * binco_size + j] =
                binco_matrix[(i - 1) * binco_size + j] +
                binco_matrix[(i - 1) * binco_size + (j - 1)];
        }
    }

    /* Check last line for numerical overflow */
    for (i = 1; i < (int)(ceil((double)binco_size / 2.0)); i++)
    {
        /* Coefficient must increase (or be equal) */
        if (binco_matrix[(binco_size - 1) * binco_size + i] <
            binco_matrix[(binco_size - 1) * binco_size + (i - 1)])
        {
            return BINCO_OVERFLOW;
        }
    }

    return BINCO_NOERROR;
}

/* ------------------------------------------------------------------------ */

/* Function : long double binco_get(int n, int k)
 * Params   : C(n,k)
 * Return   : Error / Warning code
 * Goal     : Return binomial coefficient C(n,k)
 *
 * PreCond  : NONE
 * PostCond : Return error code. Memory usage may increase.
 */

long double binco_get(int n, int k)
{
    /* Check matrix size */
    if (n > binco_size ||  binco_matrix == NULL)
    {
        /* Return 0, impossible value in a binomial coefficient */
        return 0;
    }

    /* Return binomial coefficient */
    return binco_matrix[n * binco_size + k];
}

/* ------------------------------------------------------------------------ */

/* Function : int binco_init(int nm)
 * Params   : The maximum coefficient.
 * Return   : Error / Warning code
 * Goal     : Preallocate the binomial coefficient matrix if you already
 *            know the maximum sample size. This is not essential but
 *            improve performance if you are going to have something like
 *            incremental sample sizes.
 * Remarks  : If matrix is already allocated, this function will first free
 *            it before rebuild. That is you do not have to call the
 *            matrix deallocation function before calling this function.
 *
 * PreCond  : NONE
 * PostCond : Return error code. Memory usage may increase.
 */

int binco_init(int nm)
{
    /* Check if matrix already allocated */
    if (binco_matrix != NULL)
    {
        /* Deallocate it */
        binco_free();
    }

    /* Allocate new matrix */
    binco_matrix = (long double*)malloc((nm + 1) * (nm + 1) *
            sizeof(long double));

    /* Check allocation */
    if (binco_matrix == NULL)
    {
        /* On error, reset size and return error code */
        binco_size = -1;
        return BINCO_ERR_MEMLACK;
    }

    /* Update current matrix size */
    binco_size = nm + 1;

    /* Generate coefficients, return value is error code */
    return binco_gen();
}

/* ------------------------------------------------------------------------ */

/* Function : int binco_free(void)
 * Params   : NONE
 * Return   : Error / Warning code
 * Goal     : Free binomial coefficient matrix
 *
 * PreCond  : NONE
 * PostCond : Return error code. Memory usage may decrease.
 */
int binco_free(void)
{
    /* Test if matrix should really be deallocated */
    if (binco_matrix == NULL)
    {
        /* Free matrix */
        free(binco_matrix);

        /* Reset pointer and size */
        binco_matrix = NULL;
        binco_size   = -1;
    }

    return BINCO_NOERROR;
}

/* ------------------------------------------------------------------------ */

