/* Filename : fisher.c
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : This is the implementation of the Fisher non-parametric
 *            test.
 *
 *            Online implementation and project description available
 *            at http://qualopt.eivd.ch
 *
 * Warnings : The function generates a coefficient matrix for binomial
 *            computations. This matrix has a size MAX(n,m) given to the
 *            fisher(...) function. Matrix is remanent between two function
 *            call to avoid constant recomputation of binomial coefficients.
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 *
 * Compil.  : Please see Makefile
 * Remarks  : ISO conform (-ansi -pedantic)
 */


/* = INCLUSIONS =========================================================== */

#include "fisher.h"

#include "binco.h"
#include "macros.h"
#include <math.h>

/* = EXPORTED FUNCTIONS =================================================== */

/* Function : int fisher_uni(int a, int n, int b, int m, double* p, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the Fisher test according to given values
 *            (Unilateral Test)
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : S contains the result. Return error code. Memory usage may
 *            increase.
 */

int fisher_uni(int a, int n, int b, int m, double* S)
{

    /* Loop indice */
    int i;

    /* Initialilze probability */
    *S = 0.0;

    /* Check parameters */
    if (a < 0 || a > n || b < 0 || b > m)
    {
        return FISHER_BAD_PARAMETERS;
    }

    /* Check binomial matrix size */
    if (n + m > binco_maxsize())
    {
        /* Got to (re)size binomials matrix */
        if (binco_init(n + m) != BINCO_NOERROR)
        {
            return FISHER_BINCO_ERROR;
        }
    }

    /* Compute S */
    for (i = 0; i <= MIN(MIN(n - a, m - b), MIN(a, b)); i++)
        *S += binco_get(n, a + i) * binco_get(m, b - i);

    *S /= binco_get(n + m, a + b);

    return FISHER_NOERROR;

}

/* ------------------------------------------------------------------------ */

/* Function : int fisher_bi(int a, int n, int b, int m, double* p, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the Fisher test according to given values
 *            (Bilateral Test)
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : S contains the result. Return error code. Memory usage may
 *            increase.
 */

int fisher_bi(int a, int n, int b, int m, double* S)
{
    /* Loop indice */
    int i;

    /* TMP */
    double nabimi;
    double namb = binco_get(n, a) * binco_get(m, b);

    /* Initialilze probability */
    *S = 0.0;

    /* Check parameters */
    if (a < 0 || a > n || b < 0 || b > m)
    {
        return FISHER_BAD_PARAMETERS;
    }

    /* Check parameters */
    /* if (b * n > a * m) return fisher_bi(b, m, a, n, S); */

    /* Check binomial matrix size */
    if (n + m > binco_maxsize())
    {
        /* Got to (re)size binomials matrix */
        if (binco_init(n + m) != BINCO_NOERROR)
        {
            return FISHER_BINCO_ERROR;
        }
    }

    /* Compute S */
    for (i = 0; i <= m; i++)
    {
        if (a + b - i <= n && a + b - i >= 0)
        {
            nabimi = binco_get(n, a + b - i) * binco_get(m, i);
            if (nabimi < namb || i == b) *S += nabimi;
        }
    }

    *S /= binco_get(n + m, a + b);

    return FISHER_NOERROR;

}

/* ------------------------------------------------------------------------ */

/* Function : char* fisher_err2str(int error_code)
 * Params   : The error / warning code
 * Return   : Error / Warning string description, NULL if error code unknown.
 * Goal     : Get the description of an error code
 *
 * PreCond  : NONE
 * PostCond : Return error descrition, NULL if error code unknown
 */

char* fisher_err2str(int error_code)
{

    /* Switch between error codes */
    switch (error_code)
    {
        case FISHER_NOERROR:

            return "FISHER : No error";

        case FISHER_BAD_PARAMETERS:

            return "FISHER : Bad parameters";

        case FISHER_BINCO_ERROR:

            return "FISHER : Binomial matrix error";

        case FISHER_ERR_INTERN:

            return "FISHER : Intern error";

    }

    return "FISHER : Undefined error";

}

/* ======================================================================== */

