
/* ------------------------------------------------------------------------ */

/* Filename : taillard.c
 * Author   : Philippe Waelti <philippe.waelti@eivd.ch>
 * Date     : 2003-07-04
 * Goal     : A Statistical Test for Comparing Success Rates.
 *
 *            This is the implementation of the statistical test presented
 *            in the paper 'Statistical Test for Comparing Success Rates'
 *            written by Eric D. Taillard, INA-EiVD.
 *
 *            Online implementation and project description available
 *            at http://qualopt.eivd.ch
 *
 * Warnings : The function generates a coefficient matrix for binomial
 *            computations. This matrix has a size MAX(n,m) given to the
 *            taillard(...) function. Matrix is remanent between two function
 *            call to avoid constant recomputation of binomial coefficients.
 *            Use binco_free() to free this matrix. You can also
 *            improve performance with binco_alloc(...) to generate the
 *            matrix if you already know the MAX(n,m) you will use.
 *            See "binco.h" for more details.
 *
 *            Part of QualOpt project from the HES-SO, QUALOPT-11731,
 *            Switzerland.
 *
 * Compil.  : Please see Makefile
 * Remarks  : ISO conform (-ansi -pedantic)
 */

/* ------------------------------------------------------------------------ */

#include "taillard.h"

#include "binco.h"
#include "macros.h"

#include <math.h>

/* ------------------------------------------------------------------------ */

/* epsilon for Newton-Raphson */
#define NEW_RAPH_EPSILON 1e-03

/* ------------------------------------------------------------------------ */

/* Term */
static double taillard_p(double p, int a, int n, int b, int m, int i, int j)
{
    return binco_get(n, i) * binco_get(m, j) * pow(p, i + j) *
        pow(1.0 - p, n + m - i - j);
}

/* ------------------------------------------------------------------------ */

/* Term, first derivate */
static double taillard_dp(double p, int a, int n, int b, int m, int i, int j)
{
    double nmij = (double)(n + m - i - j);

    if (i + j == 0)

        return - ((double)(n + m) * pow(1.0 - p, n + m - 1.0));

    else if (i + j == n + m)

        return ((double)(i + j) * pow(p, i + j - 1.0));

    else

        return binco_get(n, i) * binco_get(m, j) *
            (
             (double)(i + j) * pow(p, i + j - 1.0) * pow(1.0 - p, nmij) -
             pow(p, i + j) * (nmij) * pow(1.0 - p, nmij - 1.0)
            );

    return 0.0;
}

/* ------------------------------------------------------------------------ */

/* Term, second derivate */
static double taillard_ddp(double p, int a, int n, int b, int m, int i, int j)
{
    double nmij = (double)(n + m - i - j);

    if (i + j == 0)

        return (double)(n + m) * (n + m - 1.0) * pow(1.0 - p, n + m - 2.0);

    else if (i + j == n + m)

        return (double)(i + j) * (i + j - 1.0) * pow(p, i + j - 2.0);

    else if (i + j == 1)

        return binco_get(n, i) * binco_get(m, j) *
            (
             - (n + m - 1.0) * pow(1.0 - p, n + m - 2.0) -
             (n + m - 1.0) * (pow(1.0 - p, n + m - 2.0) - p * (n + m - 2.0) *
             pow(1.0 - p, n + m - 3.0))
            );

    else

        return binco_get(n, i) * binco_get(m, j) *
            (
             (double)(i + j) *
              ((i + j - 1.0) * pow(p, i + j - 2.0) * pow(1.0 - p, nmij) -
               pow(p, i + j - 1.0) * (nmij) * pow(1.0 - p, nmij - 1.0)
              )
             -
             nmij *
              ((i + j) * pow(p, i + j - 1.0) * pow(1.0 - p, nmij - 1.0) -
               pow(p, i + j) * (nmij - 1.0) * pow(1.0 - p, nmij - 2.0)
              )
            );

    return 0.0;
}

/* ------------------------------------------------------------------------ */

static double _taillard(double p, int a, int n, int b, int m,
        double (*func)(double, int, int, int, int, int, int), int twotailed)
{
    /* Indices */
    int i, j;

    /* Statistic */
    double ST = 0.0;

    for (i = 0; i <= n; i++)
    {
        for (j = 0; j <= m; j++)
        {
            if (((i >= a) && (j <= b)) ||
                    (twotailed && (i <= n - a && j >= m - b)))
            {
                ST += func(p, a, n, b, m, i, j);
            }
        }
    }

    return ST;

}

/* ------------------------------------------------------------------------ */

/* Function S(p) */
static double taillard_S(double p, int a, int n, int b, int m)
{
    return _taillard(p, a, n, b, m, taillard_p, 0);
}

/* ------------------------------------------------------------------------ */

/* Function S'(p) */
static double taillard_dS(double p, int a, int n, int b, int m)
{
    return _taillard(p, a, n, b, m, taillard_dp, 0);
}

/* ------------------------------------------------------------------------ */

/* Function S''(p) */
static double taillard_ddS(double p, int a, int n, int b, int m)
{
    return _taillard(p, a, n, b, m, taillard_ddp, 0);
}

/* ------------------------------------------------------------------------ */

/* Function T(p) */
static double taillard_T(double p, int a, int n, int b, int m)
{
    return _taillard(p, a, n, b, m, taillard_p, 1);
}

/* ------------------------------------------------------------------------ */

/* Function T'(p) */
static double taillard_dT(double p, int a, int n, int b, int m)
{
    return _taillard(p, a, n, b, m, taillard_dp, 1);
}

/* ------------------------------------------------------------------------ */

/* Function T''(p) */
static double taillard_ddT(double p, int a, int n, int b, int m)
{
    return _taillard(p, a, n, b, m, taillard_ddp, 1);
}

/* ------------------------------------------------------------------------ */

/* Newton-Raphson */
static double newton_raphson(int a, int n, int b, int m, int twosided)
{
    /* Maximum, initialized to an intuitive value */
    double p = (double)(a + b) / (double)(n + m);

    /* Value for the Newton-Raphson, last iteration */
    double p_1;

    do
    {
        /* Last iteration value */
        p_1 = p;

        /* Move to maximum */
        if (twosided)
        {
            p = p - taillard_dT(p, a, n, b, m) / taillard_ddT(p, a, n, b, m);

            /* Check if method converge */
            if (taillard_T(p, a, n, b, m) - taillard_T(p_1, a, n, b, m) < 0.0)
                return p_1;
        }
        else
        {
            p = p - taillard_dS(p, a, n, b, m) / taillard_ddS(p, a, n, b, m);

            /* Check if method converge */
            if (taillard_S(p, a, n, b, m) - taillard_S(p_1, a, n, b, m) < 0.0)
                return p_1;
        }

    } while (LABS(p - p_1) > NEW_RAPH_EPSILON);

    return p;
}

/* ------------------------------------------------------------------------ */

/* Function : int taillard_uni(int a, int n, int b, int m, double* p, double* S)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the probability S to observe 'a' successes or more in
 *            the first population (of size n) and 'b' successes or less in
 *            the second population (of size m)
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= m
 * PostCond : S contains the result. Return error code. Memory usage may
 *            increase.
 */

int taillard_uni(int a, int n, int b, int m, double* p, double* S)
{
    /* Check parameters */
    if (a < 0 || a > n || b < 0 || b > m)
    {
        return STCSR_BAD_PARAMETERS;
    }

    /* Check parameters */
    /* if (a * m < b * n) return taillard_uni(b, m, a, n, p, S); */

    /* Check binomial matrix size */
    if (MAX(n, m) > binco_maxsize())
    {
        /* Got to (re)size binomials matrix */
        if (binco_init(MAX(n, m)) != BINCO_NOERROR)
        {
            return STCSR_BINCO_ERROR;
        }
    }

    /* Special cases */
    if ((a == 0 && b == 0) || (a == n && b == m))
    {
        /* Same probability */
        *S = 0.5;
    }
    else
    {
        /* Compute 'p' that maximize S */
        *p = newton_raphson(a, n, b, m, 0);

        /* Compute S value */
        *S = taillard_S(*p, a, n, b, m);

    }

    return STCSR_NOERROR;
}

/* ------------------------------------------------------------------------ */

/* Function : int taillard_bi(int a, int n, int b, int m, double* p, double* T)
 * Params   : Let sample 1 have a 'a/n' success rate and sample 2 have a 'b/m'
 *            success rate. Result will be S (See Goal).
 * Return   : Error / Warning code.
 * Goal     : Compute the probability S to observe that the success rate
 *            of both samples are different
 * PreCond  : a >= 0 && a <= n && b >= 0 && b <= n
 * PostCond : T contains the result. Return error code. Memory usage may
 *            increase.
 */

int taillard_bi(int a, int n, int b, int m, double* p, double* T)
{
    /* Check parameters */
    if (a < 0 || a > n || b < 0 || b > m)
    {
        return STCSR_BAD_PARAMETERS;
    }

    /* Check parameters */
    /* if (a * m < b * n) return taillard_bi(b, m, a, n, p, T); */

    /* Check binomial matrix size */
    if ((n + m) > binco_maxsize())
    {
        /* Got to (re)size binomials matrix */
        if (binco_init(n + m) != BINCO_NOERROR)
        {
            return STCSR_BINCO_ERROR;
        }
    }

    /* Special cases */
    if ((a == 0 && b == 0) || (a == n && b == m))
    {
        *T = 0.0;
    }
    else
    {
        /* Compute 'p' that maximize S */
        *p = newton_raphson(a, n, b, m, 1);

        /* Compute S value */
        *T = taillard_T(*p, a, n, b, m);

    }

    return STCSR_NOERROR;
}

/* ------------------------------------------------------------------------ */

/* Function : char* taillard_err2str(int error_code)
 * Params   : The error / warning code
 * Return   : Error / Warning string description, NULL if error code unknown.
 * Goal     : Get the description of an error code
 *
 * PreCond  : NONE
 * PostCond : Return error descrition, NULL if error code unknown
 */

char* taillard_err2str(int error_code)
{

    /* Switch between error codes */
    switch (error_code)
    {
        case STCSR_NOERROR:

            return "STCSR : No error";

        case STCSR_BAD_PARAMETERS:

            return "STCSR : Bad parameters";

        case STCSR_BINCO_ERROR:

            return "STCSR : Binomial matrix error";

        case STCSR_ERR_INTERN:

            return "STCSR : Intern error";

    }

    return "STCSR : Undefined error";

}

/* ------------------------------------------------------------------------ */

