/**
 * <p> Binomial coefficients generation. </p>
 *
 * <p> Part of QualOpt project from the HES-SO, QUALOPT-11731,
 * Switzerland.</p>
 *
 * @author  Philippe Waelti / &lt;philippe.waelti at heig-vd.ch&gt;
 * @version 0.20
 * @since   0.10
 */


public class Binomials {

    /** Maximum coefficient */
    private static int maxSize = -1;

    /** Binomial coefficient matrix */
    private static double[][] binomialMatrix;

    /** No constructor */
    private Binomials() {}

    /** Generate the binomial matrix */
    private static void generateMatrix(int size) throws STATSMemoryException {

        /* Updatesize */
        maxSize = size + 1;

        /* " Free " (Hope grabage collector will do his work) the current
           matrix */
        binomialMatrix = null;

        try {
            /* Allocate new matrix */
            binomialMatrix = new double[maxSize][maxSize];
        } catch (OutOfMemoryError e) {
            maxSize = -1;
            throw new STATSMemoryException("generateMatrix");
        }

        /* Generate extremum value of Pascal triangle */
        for (int i = 0; i < maxSize; i++) {
            binomialMatrix[i][0] = binomialMatrix[i][i] = 1.0;
        }

        /* Generate Pascal triangle coefficients */
        for (int i = 2; i < maxSize; i++) {
            for (int j = 1; j <= i - 1; j++) {
                binomialMatrix[i][j] =
                    binomialMatrix[i - 1][j] + binomialMatrix[i - 1][j - 1];
            }
        }
    }

    /** Get a binomial coefficient C(n,k)
     *  @param n N binomial coefficient
     *  @param k K binomial coefficient
     *  @return The binomial coefficient C(n,k)
     *  @exception STATSMemoryException
     */
    public static double getBinomial(int n, int k)
        throws STATSMemoryException {

        /* Check matrix size */
        if (n >= maxSize) {
            generateMatrix(n);
        }

        /* Return coefficient */
        return binomialMatrix[n][k];
    }
}


