/**
 * <p>Implementation of the statistical test presented
 * in the paper 'Few statistical tests for proportions comparison'
 * published by Eric Taillard, Philippe Waelti, Jacques Zuber, EiVD,
 * 2006</p>
 *
 * <p> Online implementation and project description available
 * at <a HREF="http://qualopt.eivd.ch">project page</a>.</p>
 *
 * <p> Part of QualOpt project from the HES-SO, QUALOPT-11731,
 * Switzerland.</p>
 *
 * @author  Philippe Waelti / &lt;philippe.waelti at heig-vd.ch&gt;
 * @version 0.20
 * @since   0.10
 */

public class STATS {

/* = EXPORTED FUNCTIONS =================================================== */

    /**
     * [Taillard] Statistical test (one-tailed) to compare proportions.
     * @param a The number of success in sample 1
     * @param n The number of instances in sample 1
     * @param b The number of success in sample 2
     * @param m The number of instances in sample 2
     *
     * @return The significance level (p-value) for sample 1 to be better
     *         than sample 2
     *
     * @exception STATSException
     */
    public static double taillardOneTailed(int a, int n, int b, int m)
        throws STATSException {

        /* THE Probability */
        double s;

        /* Check arguments */
        if (a < 0 || a > n || b < 0 || b > m) {
            throw new STATSBadArgsException("compareSamples");
        }

        /* Special cases */
        if ((a == 0 && b == 0) || (a == n && b == m)) {
            return 0.5;
        }

        /* Normal case */
        return taillardS(newtonRaphson(a, n, b, m, false), a, n, b, m);
    }

    /* -------------------------------------------------------------------- */

    /**
     * [Taillard] Statistical test (two-tailed) to compare proportions.
     * @param a The number of success in sample 1
     * @param n The number of instances in sample 1
     * @param b The number of success in sample 2
     * @param m The number of instances in sample 2
     *
     * @return The significance level (p-value) for sample 1 to be different
     *         to sample 2
     *
     * @exception STATSException
     */
    public static double taillardTwoTailed(int a, int n, int b, int m)
        throws STATSException {

        /* THE Probability */
        double s;

        /* Check arguments */
        if (a < 0 || a > n || b < 0 || b > m) {
            throw new STATSBadArgsException("compareSamples");
        }

        /* Special cases */
        if ((a == 0 && b == 0) || (a == n && b == m)) {
            return 0.0;
        }

        /* Normal case */
        return taillardT(newtonRaphson(a, n, b, m, true), a, n, b, m);
    }

    /* -------------------------------------------------------------------- */

    /**
     * Fisher test (one-tailed) to compare proportions.
     * @param a The number of success in sample 1
     * @param n The number of instances in sample 1
     * @param b The number of success in sample 2
     * @param m The number of instances in sample 2
     *
     * @return The significance level (p-value) for sample 1 to be better
     *         than sample 2
     *
     * @exception STATSException
     */
    public static double fisherOneTailed(int a, int n, int b, int m)
        throws STATSException {

        /* THE Probability */
        double s = 0.0;

        /* Check arguments */
        if (a < 0 || a > n || b < 0 || b > m) {
            throw new STATSBadArgsException("compareSamples");
        }

        /* Compute S */
        for (int i = 0; i <= Math.min(Math.min(n - a, m - b),
                    Math.min(a, b)); i++)
            s += Binomials.getBinomial(n, a + i) *
                Binomials.getBinomial(m, b - i);

        return s /= Binomials.getBinomial(n + m, a + b);
    }

    /* -------------------------------------------------------------------- */

    /**
     * Fisher test (two-tailed) to compare proportions.
     * @param a The number of success in sample 1
     * @param n The number of instances in sample 1
     * @param b The number of success in sample 2
     * @param m The number of instances in sample 2
     *
     * @return The significance level (p-value) for sample 1 to be different
     *         to sample 2
     *
     * @exception STATSException
     */
    public static double fisherTwoTailed(int a, int n, int b, int m)
        throws STATSException {

        /* THE Probability */
        double s = 0.0;

        /* Invariant */
        double nabimi;
        double namb = Binomials.getBinomial(n, a) *
            Binomials.getBinomial(m, b);

        /* Compute S */
        for (int i = 0; i <= m; i++)
        {
            if (a + b - i <= n && a + b - i >= 0)
            {
                nabimi = Binomials.getBinomial(n, a + b - i) *
                    Binomials.getBinomial(m, i);
                if (nabimi < namb || i == b) s += nabimi;
            }
        }

        return s /= Binomials.getBinomial(n + m, a + b);
    }

/* = CONSTANTS ============================================================ */

    /* Epsilon for Newton-Raphson */
    private static final double EPSILON = 1e-03;

/* = NOT EXPORTED FUNCTIONS =============================================== */

    /* The constructor, in fact no constructor... */
    private STATS() {}

    /* -------------------------------------------------------------------- */

    /* Term */
    private static double taillardP(double p, int a, int n, int b, int m,
            int i, int j) throws STATSException
    {
        return Binomials.getBinomial(n, i) * Binomials.getBinomial(m, j) *
            Math.pow(p, i + j) * Math.pow(1.0 - p, n + m - i - j);
    }

    /* -------------------------------------------------------------------- */

    /* Term, first derivate */
    private static double taillardDP(double p, int a, int n, int b, int m,
            int i, int j) throws STATSException
    {
        double nmij = (double)(n + m - i - j);

        return Binomials.getBinomial(n, i) * Binomials.getBinomial(m, j) *
            ((double)(i + j) * Math.pow(p, i + j - 1.0) *
             Math.pow(1.0 - p, nmij) - Math.pow(p, i + j) * (nmij) *
             Math.pow(1.0 - p, nmij - 1.0));
    }

    /* -------------------------------------------------------------------- */

    /* Term, second derivate */
    private static double taillardDDP(double p, int a, int n, int b, int m,
            int i, int j) throws STATSException
    {
        double nmij = (double)(n + m - i - j);

        return Binomials.getBinomial(n, i) * Binomials.getBinomial(m, j) *
            ((double)(i + j) * ((i + j - 1.0) * Math.pow(p, i + j - 2.0) *
             Math.pow(1.0 - p, nmij) - Math.pow(p, i + j - 1.0) * (nmij) *
             Math.pow(1.0 - p, nmij - 1.0)) - nmij * ((i + j) *
             Math.pow(p, i + j - 1.0) * Math.pow(1.0 - p, nmij - 1.0) -
             Math.pow(p, i + j) * (nmij - 1.0) *
             Math.pow(1.0 - p, nmij - 2.0)));
    }

    /* -------------------------------------------------------------------- */

    private static double _taillard(double p, int a, int n, int b, int m,
            int level, boolean twotailed) throws STATSException
    {
        /* Indices */
        int i, j;

        /* Statistic */
        double ST = 0.0;

        for (i = 0; i <= n; i++)
        {
            for (j = 0; j <= m; j++)
            {
                if (((i >= a) && (j <= b)) ||
                        (twotailed && (i <= n - a && j >= m - b)))
                {
                    if (level == 0)
                        ST += taillardP(p, a, n, b, m, i, j);
                    else if (level == 1)
                        ST += taillardDP(p, a, n, b, m, i, j);
                    else
                        ST += taillardDDP(p, a, n, b, m, i, j);
                }
            }
        }

        return ST;
    }

    /* -------------------------------------------------------------------- */

    /* Function S(p) */
    private static double taillardS(double p, int a, int n, int b, int m)
        throws STATSException
    {
        return _taillard(p, a, n, b, m, 0, false);
    }

    /* Function S'(p) */
    private static double taillardDS(double p, int a, int n, int b, int m)
        throws STATSException
    {
        return _taillard(p, a, n, b, m, 1, false);
    }

    /* Function S''(p) */
    private static double taillardDDS(double p, int a, int n, int b, int m)
        throws STATSException
    {
        return _taillard(p, a, n, b, m, 2, false);
    }

    /* Function T(p) */
    private static double taillardT(double p, int a, int n, int b, int m)
        throws STATSException
    {
        return _taillard(p, a, n, b, m, 0, true);
    }

    /* Function T'(p) */
    private static double taillardDT(double p, int a, int n, int b, int m)
        throws STATSException
    {
        return _taillard(p, a, n, b, m, 1, true);
    }

    /* Function T''(p) */
    private static double taillardDDT(double p, int a, int n, int b, int m)
        throws STATSException
    {
        return _taillard(p, a, n, b, m, 2, true);
    }


/* ------------------------------------------------------------------------ */

    /* Newton-Raphson maximization */
    private static double newtonRaphson(int a, int n, int b, int m,
            boolean twotailed) throws STATSException {

        /* Templates values of function */
        double p = (double)(a + b) / (double)(n + m);
        double p_1;

        /* While epsilon is not reached and method converge */
        do {

            p_1 = p;

            if (twotailed)
            {
                p = p - taillardDT(p, a, n, b, m) / taillardDDT(p, a, n, b, m);

                if (taillardT(p, a, n, b, m) - taillardT(p_1, a, n, b, m) < 0.0)
                    return p_1;

            } else {

                p = p - taillardDS(p, a, n, b, m) / taillardDDS(p, a, n, b, m);

                if (taillardS(p, a, n, b, m) - taillardS(p_1, a, n, b, m) < 0.0)
                    return p_1;
            }

        } while (Math.abs(p - p_1) > EPSILON);

        return p;
    }

/* ------------------------------------------------------------------------ */

}

